<?php
require_once 'constants.php';
    class Token {
    static function Sign($payload, $key, $expire = 60*60) {
        // Ensure payload is an array
        if (!is_array($payload)) {
            return json_encode([
                'headers' => [
                    'status' => 'failed',
                    'responseCode' => 400,
                    'message' => 'Invalid payload format, expected array'
                ]
            ]);
        }

        // Check if a token exists in the session
        if (isset($_SESSION['token'])) {
            // Verify the existing token
            $verificationResult = json_decode(self::Verify($_SESSION['token'], $key), true);

            // If token is still valid, return it
            if ($verificationResult['headers']['responseCode'] === 200) {
                return json_encode(
                    [
                        'headers' => [
                            'status' => 'success',
                            'responseCode' => 200,
                            'message' => 'token already generated'
                        ],
                        'body' => [
                            'token' => $_SESSION['token']
                        ]      
                    ]
                );
            }
        }

        // Generate a new token if not valid or no token exists
        $headers = ['algo' => 'HS256', 'type' => 'JWT'];
        if ($expire) {
            $headers['expire'] = time() + $expire;
        }
        $headers_encoded = base64_encode(json_encode($headers));

        // Payload
        $payload['time'] = time();  // Ensure payload is an array
        $payload_encoded = base64_encode(json_encode($payload));

        // Signature
        $signature = hash_hmac('SHA256', $headers_encoded . $payload_encoded, $key);
        $signature_encoded = base64_encode($signature);

        // Create the token
        $newToken = $headers_encoded . '.' . $payload_encoded . '.' . $signature_encoded;

        // Store the new token in the session
        $_SESSION['token'] = $newToken;
        setcookie(
            'session_token', 
            $newToken, 
            time() + 60 * 30,  // 1 hour expiration
            '/', 
            '', 
            true,  // Secure: Only sent over HTTPS
            true   // HttpOnly: Not accessible via JavaScript
        );

        return json_encode(
            [
                'headers' => [
                    'status' => 'success',
                    'responseCode' => 200,
                    'message' => 'token generated'
                ],
                'body' => [
                    'token' => $newToken
                ]      
            ]
        );
    }


    static function Verify($token, $key) {
        // Break token parts
        $token_parts = explode('.', $token);

        // Verify if token has 3 parts
        if (count($token_parts) !== 3) {

            return json_encode(
                [
                    'headers' => [
                        'status' => 'failed',
                        'responseCode' => 400,
                        'message' => 'invalid token'
                    ],      
                ]
            );
        }

        // Verify Signature
        $signature = base64_encode(hash_hmac('SHA256', $token_parts[0] . $token_parts[1], $key));
        if ($signature !== $token_parts[2]) {
            return json_encode(
                [
                    'headers' => [
                        'status' => 'failed',
                        'responseCode' => 401,
                        'message' => 'invalid token signature'
                    ],      
                ]
            );
        }

        // Decode headers & payload
        $headers = json_decode(base64_decode($token_parts[0]), true);
        $payload = json_decode(base64_decode($token_parts[1]), true);

        // Verify token expiration
        if (isset($headers['expire']) && $headers['expire'] < time()) {
            return json_encode(
                [
                    'headers' => [
                        'status' => 'failed',
                        'responseCode' => SESSION_TIME_EXCEEDED,
                        'message' => 'session time exceeded'
                    ],      
                ]
            );
        }

        // If token is successfully verified
        return json_encode(
            [
                'headers' => [
                    'status' => 'success',
                    'responseCode' => 200,
                    'message' => 'valid token'
                ],      
            ]
        );
    }
}
